/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* ALI M1543C Southbridge access routines */
/* (c) 1999 Alpha Processor Inc */
/* Begun 11 Feb 1999 by Stig Telfer, API */

#define TRACE_ENABLE		/* enable this for verbose southbridge debug */

#include "lib.h"
#include "uilib.h"
#include "info.h"

#include "platform.h"
#include "northbridge.h"
#include "southbridge.h"

#include "cmos_rtc.h"
#include "tabledriver.h"


#ifdef USE_PMU
static void ali_acpi_cfg( void );
#endif



/*----------------------------------------------------------------------*/
/* IO Operation tables */

static const TblArray set_up_sb[] = {

    CFGWB(   0, PLAT_ISAID, 0, ISA_PIC, 0x67 ),		/* some optimisations */
    CFGSETB( 0, PLAT_ISAID, 0, ISA_ISAC1, 0x48 ),	/* 32-bit DMA */
    CFGSETB( 0, PLAT_ISAID, 0, ISA_ISAC2, 0x80 ),	/* port 92 reset */

    /* No interrupt lines are connected */

    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT1, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT2, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT3, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT4, 0x00 ),

    /* Disable the USB, AGP, SMBus, PMU & IDE IRQ routings */

    CFGWB( 0, PLAT_ISAID, 0, ISA_USBIR, 0x80 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_AGPIS, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_SMBIR, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_SCIIR, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_IDENR1, 0x00 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_IDENR2, 0x00 ),

    /* All non-ISA interrupt sources should now be disabled */

    /* Configure System Management Interrupt (SMI) */

#if 1
    CFGWB( 0, PLAT_PMUID, 0, PMU_SMI_CNTL, SMI_CNTL_SMI_ENA ),
#else
    CFGWB( 0, PLAT_PMUID, 0, PMU_SMI_CNTL, 0x00 ),
#endif

    /* initialise the 8259 controllers via PC-IO space */

    IOWB( 0x20, 0x11 ),
    IOWB( 0x21, 0x00 ),                 /* IRQ0->7 are vectors 0->7 */
    IOWB( 0x21, 0x04 ),                 /* slave on IRQ 2 */
    IOWB( 0x21, 0x01 ),                 /* use x86 mode */

    IOWB( 0xA0, 0x11 ),
    IOWB( 0xA1, 0x08 ),                 /* IRQ8->15 are vectors 8->15 */
    IOWB( 0xA1, 0x02 ),                 /* slave on IRQ 2 */
    IOWB( 0xA1, 0x01 ),                 /* use x86 mode */

    /* disable all ISA interrupts */

    IOWB( 0x21, 0xFF ),                 /* ints 0-7 active low */
    IOWB( 0xA1, 0xFF ),			/* ints 8-F active low */

    /* Configure system programmable timer */

    IOWB( 0x43, 0x30 ),         /* timer 0 into mode 0 */
    IOWB( 0x40, 0x00 ),         /* send LSB of count */
    IOWB( 0x40, 0x00 ),         /* send MSB of count */

    OP_EOT
};


#ifdef USE_IDE
static const TblArray set_up_ide[] = {		/* Soohoon's setup code */

    CFGWB( 0, PLAT_ISAID, 0, ISA_IDEIC, 0x4D ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT1, 0x31 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PIRT2, 0xB9 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_PILET, 0x0F ),
    CFGWB( 0, PLAT_IDEID, 0, IDE_CMD, 0x05 ),

    /* writes to the northbridge's PCI configuration space */

    CFGWB( 0, 0, 0, 0x04, 0x06 ),
    CFGWB( 0, 0, 0, 0x5A, 0x07 ),
    CFGWB( 0, 0, 0, 0x56, 0xCC ),

    CFGWL( 0, PLAT_IDEID, 0, IDE_BA5, 0x0000F000 ),

    OP_EOT
};
#endif



#ifdef USE_PMU
static const TblArray set_up_pmu_acpi[] = {

    /* configure STPCLK break events */
    CFGWB( 0, PLAT_PMUID, 0, PMU_BES1,	0x12 ),
    CFGWB( 0, PLAT_PMUID, 0, PMU_BES1,	0x0 ),

    /* program ISA bridge to enable stopping of various device clocks */
    CFGWW( 0, PLAT_ISAID, 0, ISA_SMCC1,	0xF0E0 ),
    CFGWB( 0, PLAT_ISAID, 0, ISA_SCIIR,	0x81 ),	/* SCI: to IRQ9 */

    /* configure PMU */
    CFGWB( 0, PLAT_PMUID, 0, PMU_ATPC,	0x4F ),
    CFGWW( 0, PLAT_PMUID, 0, PMU_PTS,	0x0A69 ),
    CFGWB( 0, PLAT_PMUID, 0, PMU_PIIGS,	0x06 ),
    CFGWB( 0, PLAT_PMUID, 0, PMU_BES1,	0x12 ),	/* break events */
    CFGWB( 0, PLAT_PMUID, 0, PMU_PCCA,	0x04 ),
    CFGWB( 0, PLAT_PMUID, 0, PMU_DSC,	0x40 ),

    OP_EOT
};

#endif

/*----------------------------------------------------------------------*/
/* GPIO pin cache widgetry.  Every application using the GPIO bits must
 * use this cached copy and keep the cached copy up to date, or strange
 * things may happen...
 */

uint8 ali_gpio_data=0, ali_gpio_dirn=0;
uint8 ali_gpio_cached = FALSE;

void ali_load_gpio_cache( void )
{
    ali_gpio_dirn = pcicfgrb( 0, PLAT_PMUID, 0, PMU_DCGPIO );
    ali_gpio_data = pcicfgrb( 0, PLAT_PMUID, 0, PMU_DOGPIO );
    ali_gpio_cached = TRUE;
}

/*----------------------------------------------------------------------*/
/* Initialisation of ISA Bus and other devices hanging off a typical 
 * southbridge */

void sb_setup( void )
{
#ifdef USE_IDE
    TRACE( "Running IDE table\n");
    TblWrite( set_up_ide );	/* Enable the M5229 IDE interface */
#endif

    TRACE("Running set_up_sb\n" );
    TblWrite( set_up_sb );


    /* now do some device initialisations */

    diags_subsys_stat( SYS_ISA, DEV_PROBING );

    TRACE("Running AcerSIOInit\n" );
    AcerSIOInit();	/* SuperIO devices (embedded within Southbridge) */
    InitRTC();		/* Real time clock setup */

    diags_subsys_stat( SYS_ISA, DEV_SETUP );
}



/*----------------------------------------------------------------------*/
/* Fixups required during initialisation process */
void sb_fixup( void )
{
    unsigned char val; 
    int alirev;

    mobo_logf( LOG_INFO "Enabling Southbridge NMI...\n");

    val = inportb( 0x61 );                        /* NMI ctrl/status port */
    val |= 0x0C;
    outportb( 0x61, val );                        /* write 1's to clear NMIs */
    val &= ~0x0C;
    outportb( 0x61, val );                        /* write 0's to re-enable */

    /* Enable PCI bus parity check -> NMI */
    val = pcicfgrb( 0, PLAT_ISAID, 0, ISA_PIPM );
    val |= 1;
    pcicfgwb( 0, PLAT_ISAID, 0, ISA_PIPM, val );


#ifdef USE_PMU
    /* configure the power management unit */
    ali_acpi_cfg();
#endif

    ali_load_gpio_cache();


    /* Register information about this chip */
    alirev = pcicfgrb(0, PLAT_ISAID, 0, ISA_REV );
    info_submit( SRC_CHIPSET, SUBJ_CHIPSET, "ALI Southbridge Rev",
		"Revision %02X", alirev );
}


/*----------------------------------------------------------------------*/
/* ACPI configuration routine */

#ifdef USE_PMU
static void ali_acpi_cfg( void )
{
    unsigned b;

    /* configuration via ACPI according to the ALI 1543C ref p292 */
    mobo_logf( LOG_INFO "Configuring ACPI...\n" );

    /* ACPI BIOS initialise chipset registers: program ACPI/SMB IO addrs */

#if 0	/* PCI bus configuration has allocated resources */

    b = pcicfgrl( 0, PLAT_PMUID, 0x0, PMU_ABA ) & ~0x1U;
    mobo_logf( LOG_WARN "ACPI: base addr is 0x%X\n", b);

#else	/* choose base addresses ourselves */

    b = 0xDF00U;
    pcicfgwl( 0, PLAT_PMUID, 0x0, PMU_ABA, b );		/* PMU base = 0xDF00 */
    pcicfgwl( 0, PLAT_PMUID, 0x0, PMU_SBA, b + 0x80 ); 	/* SMB base = 0xDF80 */

    pcicfgwb( 0, PLAT_PMUID, 0x0, PMU_CMD, 0x01 );	/* enable IO response */

    b = pcicfgrl( 0, PLAT_PMUID, 0x0, PMU_ABA ) & ~0x1U;
    mobo_logf( LOG_WARN "ACPI: base addr is 0x%X\n", b);
#endif


    mobo_logf( LOG_WARN "writing ACPI config table\n" );
    TblWrite( set_up_pmu_acpi );			/* guts of config op */


    mobo_logf( LOG_WARN "Configuring ACPI in IO space\n" );

    /* enable ACPI clock control function */
    outportb( b + 0x11, 0x02 );

    /* Initialise ACPI features - see ALI doc p193 */

    outportw( b + 0x00, 0xFFFFU );	/* clear PM1_STS */
    outportw( b + 0x02, 0x0000U );	/* disable PM1_EN */

    outportw( b + 0x18, 0xFFFFU );	/* clear GPE0_STS */
    outportw( b + 0x1A, 0x0000U );	/* disable GPE0_EN */

    outportw( b + 0x1C, 0xFFFFU );	/* clear GPE1_STS */
    outportw( b + 0x1E, 0x0000U );	/* disable GPE1_EN */

}

#endif 

/*----------------------------------------------------------------------*/
/* ALI specific interface routines */

String ali_revstr( String buf )
{
    int alirev = pcicfgrb(0, PLAT_ISAID, 0, ISA_REV );
    sprintf_dbm( buf, "Revision %02X", alirev );
    return buf;
}



/*----------------------------------------------------------------------*/
/* I2C and System Management Bus support */

#define ALI_BUGGY_DELAY		100		/* breather period in usec */

/* Operations relating to the IO-space-mapped SMBus registers */
static uint32 smb_base_reg = 0;
#define SMB_READ( reg )		inportb( smb_base_reg + (reg) )
#define SMB_WRITE( reg, val )	outportb( smb_base_reg + (reg), val )

static void i2c_error_analyse( const uint8 status );
static void i2c_error_clear( void );

static DBM_STATUS i2c_clr_smi( void )
{
    uint8 smicntl;
    int status, newstatus;
    int rval, newrval;
    DBM_STATUS sval = STATUS_SUCCESS;

    smicntl = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMI_CNTL );
    TRACE( "SMI: Control register is 0x%02X\n", smicntl );


    /* Look at the SMI status, particularly for SMBUS */
    status = pcicfgrl( 0, PLAT_PMUID, 0, PMU_SSWS );
    if( status & SSWS_SMB )
    {
	/* Start an SMI ACK cycle */
	pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMI_CNTL, smicntl | SMI_CNTL_SMIACK );

        rval = SMB_READ( SMBUS_SMBSTS );
	if ( rval & SMBUS_SMBSTS_ERROR )
	{
	    i2c_error_analyse( rval );
	    sval = STATUS_WARNING;
	}
        SMB_WRITE( SMBUS_SMBSTS, rval );	/* W1C the SMBSTS register */
        newrval = SMB_READ( SMBUS_SMBSTS );
        TRACE( "SMI: SMBus status is 0x%02X -> 0x%02X\n", rval, newrval );

	/* Finish SMI ACK cycle */
	pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMI_CNTL, smicntl );
    }

    pcicfgwl( 0, PLAT_PMUID, 0, PMU_SSWS, SSWS_SMB );
    newstatus = pcicfgrl( 0, PLAT_PMUID, 0, PMU_SSWS );

    TRACE( "SMI: SSWS is 0x%06lX -> 0x%06lX\n", status, newstatus );

    if ( newstatus & SSWS_SMB )
    {
	mobo_logf( LOG_WARN "SMI: unable to clear SMB request 0x%06X->0x%06X\n",
		status, newstatus );
	sval = STATUS_FAILURE;
    }

    return sval;
}


#define SMB_POLL_DELAY		50		/* in usec */
#define SMB_POLL_TIMEOUT	1000000		/* in usec */

static DBM_STATUS i2c_wait_smi( void )
{
    int status;
    int timeout = 0;

    TRACE( "Waiting for SMI...\n" );

    do {
	status = pcicfgrl( 0, PLAT_PMUID, 0, PMU_SSWS );

	usleep( SMB_POLL_DELAY );
	timeout += SMB_POLL_DELAY;

	if( timeout >= SMB_POLL_TIMEOUT )
	{
	    mobo_logf( LOG_CRIT "I2C: not raising anticipated SMI!\n" );
	    return STATUS_FAILURE;
	}

    } while ( (status & SSWS_SMB) == 0 );

    return STATUS_SUCCESS;
}


static void i2c_error_analyse( const uint8 status )
{
    int i;
    static const String interp[] = {
	"Host Slave Busy (slave is going to receive a command)",
	"Host Slave Status (SMI raised by slave interface)",
	"SMBus is at idle status",
	"Host Busy (SMBus host controller is completing cmd)",
	"SMI asserted by SMBus controller (after command)",
	"SMI asserted due to the generation of an error on I2C bus",
	"I2C Bus collision occurred (or no acknowledge from target)",
	"Terminated bus transaction in response to ABORT command"
    };

    mobo_logf( LOG_CRIT "ALI M1543C SMBus error analysis (0x%02X):\n",
		status );

    for ( i=0; i<8; i++ )
    {
	if ( status & (1<<i) )
	    mobo_logf( LOG_CRIT "  %s\n", interp[ i ] );
    }
}


/* Wait until the I2C bus is idle */

#define TIMEOUTVAL 100000			/* in usec */

static DBM_STATUS i2c_wait_ready( void )
{
    int n;
    int status, lastStatus;

    /* First, clear old status bits */
    SMB_WRITE( SMBUS_SMBSTS, 0xFF );

    /* Wait for Bus Idle. */
    for ( n=0, status=0, lastStatus=-1; n < TIMEOUTVAL; n+= ALI_BUGGY_DELAY )
    {
	/* Read status. */
	status = SMB_READ( SMBUS_SMBSTS );

	/* Debug */
	if (status != lastStatus)
	{
	    lastStatus = status;
	}

	if ( status & SMBUS_SMBSTS_IDL_STS )
		return STATUS_SUCCESS;

	/* DEBUG: ALI SMBus controller needs a breather? */
	usleep( ALI_BUGGY_DELAY );
    }

    /* Timeout condition */
    mobo_logf( LOG_CRIT "ALI M1543C SMBus is locked up busy.\n");
    i2c_error_analyse( status );
    return STATUS_FAILURE;
}



/* Wait until a transaction has been completed */

static DBM_STATUS i2c_wait_complete( void ) 
{
    int n;
    uint8 status, lastStatus;

    for ( n=0, status=0, lastStatus=0xFF; n < TIMEOUTVAL; n += ALI_BUGGY_DELAY )
    {
        /* Read status. */
        status = SMB_READ( SMBUS_SMBSTS );

	/* DEBUG - ALI controller may need short breather before ready? */
	usleep( ALI_BUGGY_DELAY );

#if 0
        /* Debug */
        if (status != lastStatus)
        {
            lastStatus = status;
            TRACE("status=%x\n", status);
        }
#endif

	/* Check for presence of any errors */
	if ( status & SMBUS_SMBSTS_ERROR )
	{
	    i2c_error_analyse( status );
	    return STATUS_FAILURE;
	}

        if ( status & SMBUS_SMBSTS_SMI_I_STS )	/* completed successfully */
	{
	    SMB_WRITE( SMBUS_SMBSTS, 0xFF );
	    return STATUS_SUCCESS;
	}
    }

    /* Timeout condition */
    mobo_logf( LOG_CRIT "ALI M1543C SMBus timed out on transaction.\n");
    i2c_error_analyse( status );
    return STATUS_FAILURE;
}



/*------------------------------------------------------------------------*/
/* I2C byte/block read/write transactions */

static DBM_STATUS i2c_byte_write( const uint8 addr, const int cmd,
					const uint8 *data )
{
    DBM_STATUS sval;
    uint8 status;

    TRACE( "Running on address 0x%02X command 0x%02X\n", addr, cmd );

    /* Clear the SMBus status and wait until idle */
    sval = i2c_wait_ready( );
    if( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT
		"I2C: startup failure on byte write transaction!\n" );
	i2c_error_analyse( SMB_READ( SMBUS_SMBSTS ) );
	return STATUS_FAILURE;
    }

    /* Write the device address */
    SMB_WRITE( SMBUS_ADDR, addr ); 

    /* Set the command byte to be written (if appropriate) */
    if( cmd >= 0 )
    {
	/* Write the command - byte write */
	SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_WRRDB );

	/* Set the data to be written */
	SMB_WRITE( SMBUS_DATA_A, *data );

	SMB_WRITE( SMBUS_CMD, cmd );
    }
    else
    {
	/* Quick command means writing without sending a command byte */
	SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_SNDRCV );

	/* Set the data to be written, it goes in the first byte ie cmd byte */
	SMB_WRITE( SMBUS_CMD, *data );
    }

    /* Set the start condition */
    SMB_WRITE( SMBUS_START, 0xFF );

    /* Wait for an SMI */
    sval = i2c_wait_smi( );
    if( sval != STATUS_SUCCESS )
    {
        mobo_logf( LOG_CRIT
                "I2C: completion failure on byte write transaction!\n" );
        i2c_error_analyse( SMB_READ( SMBUS_SMBSTS ) );
        return STATUS_FAILURE;
    }

    /* Was the transaction successful? */
    status = SMB_READ( SMBUS_SMBSTS );
    i2c_clr_smi( );
    return (status & SMBUS_SMBSTS_SMI_I_STS) ? STATUS_SUCCESS : STATUS_FAILURE;
}


static DBM_STATUS i2c_byte_read( const uint8 addr, const int cmd, uint8 *data )
{
    DBM_STATUS sval;
    uint8 status;

    TRACE( "Running on address 0x%02X command 0x%02X\n", addr, cmd );

    /* Clear the SMBus status and wait until idle */
    sval = i2c_wait_ready( );
    if( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT
		"I2C: startup failure on byte read transaction!\n" );
	i2c_error_analyse( SMB_READ( SMBUS_SMBSTS ) );
	return STATUS_FAILURE;
    }

    /* Write the device address, setting the bit indicating a read */
    SMB_WRITE( SMBUS_ADDR, addr | 0x01 ); 

    /* Set the command byte to be written (if appropriate) */
    if( cmd >= 0 )
    {
	/* Write the command - byte write */
	SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_WRRDB );

	SMB_WRITE( SMBUS_CMD, cmd );
    }
    else
    {
	/* Quick command means writing without sending a command byte */
	SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_SNDRCV );
    }

    /* Set the start condition */
    SMB_WRITE( SMBUS_START, 0xFF );

    /* Wait for an SMI */
    sval = i2c_wait_smi( );
    if( sval != STATUS_SUCCESS )
    {
        mobo_logf( LOG_CRIT
                "I2C: completion failure on byte read transaction!\n" );
        i2c_error_analyse( SMB_READ( SMBUS_SMBSTS ) );
        return STATUS_FAILURE;
    }

    /* Was the transaction successful? */
    status = SMB_READ( SMBUS_SMBSTS );
    if( (status & SMBUS_SMBSTS_SMI_I_STS) == 0 )
    {
	mobo_logf( LOG_WARN
		"I2C: completion error on byte read transaction\n" );
	return STATUS_FAILURE;
    }

    i2c_clr_smi( );

    TRACE( "byte read transaction completed successfully\n" );

    /* Read the byte back from the data register */
    *data = SMB_READ( SMBUS_DATA_A );
    TRACE( "got byte 0x%02X\n", *data );
    return STATUS_SUCCESS;
}



static DBM_STATUS i2c_block_write( const uint8 addr, const int cmd, 
					const size_t nbytes, const uint8 *data )
{
    DBM_STATUS sval;
    int count;
    uint8 status;

    TRACE( "Running on %d bytes address 0x%02X command 0x%02X\n",
		nbytes, addr, cmd );

    /* Clear the SMBus status register and wait until idle */
    sval = i2c_wait_ready( );
    if( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT
		"I2C: startup failure on block write transaction!\n" );
	i2c_error_analyse( SMB_READ( SMBUS_SMBSTS ) );
	return STATUS_FAILURE;
    }

    /* Set the target device address */
    SMB_WRITE( SMBUS_ADDR, addr );

    /* Set the block write command (and clear the block pointer) */
    SMB_WRITE( SMBUS_SMBCMD, (SMBUS_SMBCMD_BLK_CLR | SMBUS_COMMAND_BLOCK) );

    /* Write the block size (max 32) to dataA */
    SMB_WRITE( SMBUS_DATA_A, nbytes );

    /* Write the bytes individually to the block data register */
    for( count = 0; count < nbytes; count++ )
    {
	SMB_WRITE( SMBUS_BLOCK, data[count] );
    }

    /* Write the command byte to the command register (if apt) */
    if ( cmd >= 0 )
    {
	SMB_WRITE( SMBUS_CMD, cmd );
    }

    /* Set the start condition */
    SMB_WRITE( SMBUS_START, 0xFF );

    /* Wait for an SMI */
    sval = i2c_wait_smi( );
    if( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT
		"I2C: completion failure on block write transaction!\n" );
	i2c_error_analyse( SMB_READ( SMBUS_SMBSTS ) );
	return STATUS_FAILURE;
    }
    i2c_clr_smi( );

    /* Was the transaction successful? */
    status = SMB_READ( SMBUS_SMBSTS );
    return (status & SMBUS_SMBSTS_SMI_I_STS) ? STATUS_SUCCESS : STATUS_FAILURE;
}



static DBM_STATUS i2c_block_read( const uint8 addr, const int cmd, 
					const size_t nbytes, uint8 *data )
{
    DBM_STATUS sval;
    int count;
    uint8 status;

    TRACE( "Running on %d bytes address 0x%02X command 0x%02X\n",
		nbytes, addr, cmd );

    /* Clear the SMBus status register and wait until idle */
    sval = i2c_wait_ready( );
    if( sval != STATUS_SUCCESS )
    {
	mobo_logf( LOG_CRIT
		"I2C: startup failure on block read transaction!\n" );
	i2c_error_analyse( SMB_READ( SMBUS_SMBSTS ) );
	return STATUS_FAILURE;
    }

    /* Set the target device address, nb set 1 for reads */
    SMB_WRITE( SMBUS_ADDR, addr | 1 );

    /* Set the block write command (and clear the block pointer) */
    SMB_WRITE( SMBUS_SMBCMD, (SMBUS_SMBCMD_BLK_CLR | SMBUS_COMMAND_BLOCK) );

    /* Write the block size (max 32) to dataA */
    SMB_WRITE( SMBUS_DATA_A, nbytes );


    /* Write the command byte to the command register (if apt) */
    if ( cmd >= 0 )
    {
	SMB_WRITE( SMBUS_CMD, cmd );
    } else {
	mobo_logf( LOG_WARN "ALI: unsupported cmd-less block access\n" );
    }

    /* Set the start condition */
    SMB_WRITE( SMBUS_START, 0xFF );

    /* First wait for an SMI (for each byte?) */
    i2c_wait_smi( );
    status = SMB_READ( SMBUS_SMBSTS );
    if( status & SMBUS_SMBSTS_ERROR )
    {
	mobo_logf( LOG_WARN
		"I2C: completion error on block write transaction\n" );
	return STATUS_FAILURE;
    }

    /* Read the bytes individually from the block data register */
    for( count = 0; count < nbytes; count++ )
    {
	data[count] = SMB_READ( SMBUS_BLOCK );
	TRACE( "read byte %d as 0x%02X\n", count, data[count] );
    }
    i2c_clr_smi( );

    return STATUS_SUCCESS;
}


/*------------------------------------------------------------------------*/
/* ALi M1543C SMBus interface routines */

/* The ALI M1543C southbridge supports one i2c bus interface.  No knowledge
 * of device target addresses, wiring, multiplexers etc is in here.  Just the
 * reading and writing of a sequence of bytes...
 */

DBM_STATUS i2cdev_init( void )
{
    uint16 val16;

    TRACE( "Looking at ALI SMB config...\n");

    /* Enable the PMU to respond to IO space transactions */
    /* NOTE: should I do this and what if the PMU doesn't respond to IO? */
    val16 = pcicfgrw( 0, PLAT_PMUID, 0, PMU_CMD );
    if ( (val16 & 1) == 0 )
    {
	mobo_logf( LOG_CRIT "PMU was not enabled for IO transactions!\n" );
	val16 |= 1;
	pcicfgww( 0, PLAT_PMUID, 0, PMU_CMD, val16 );
    }


    /* Set speed to 37.5K with IDLE detect as BaseClk *128 (NB conservative). */
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHCBC,
		PMU_SMBHCBC_37K | PMU_SMBHCBC_DL128 );

    /* Enable SMB Bus Controller */
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHSI, PMU_SMBHSI_HOST_EN );


    smb_base_reg = pcicfgrl( 0, PLAT_PMUID, 0, PMU_SBA ) & ~0x3U;
    TRACE("smb_base_reg=%x\n", smb_base_reg);
    SMB_WRITE( SMBUS_SMBSTS, 0xFF );

    return STATUS_SUCCESS;
} 




/* node = I2C bus address of target
 * cmd = goes into SMBus command byte, often used by targets as a reg index
 * 	(set to -1 if not required)
 * bytes = number of bytes to be read from target
 * data = storage of read data
 */

#define MAX_RETRY_COUNT	4

DBM_STATUS i2cdev_read( const uint8 node, const int cmd,
			const size_t bytes, uint8 *data)
{
    int bytes_remaining, retries;
    uint8 *thisdata;
    int thiscmd;
    int thisbytes;
    DBM_STATUS sval;

    TRACE( "I2C bus read of %d bytes from node %x cmd %02x\n",
		bytes, node, cmd  );

    /* Paranoia: poison the buffer so we can better spot non-reads */
    memset( data, 0xFFU, bytes );


    bytes_remaining = bytes;
    thisdata = data;
    thiscmd = cmd;
    thisbytes = 0;
    retries = 0;

    while( bytes_remaining > 0 )
    {
#if 0
	if( bytes_remaining == 1 )
	{
	    thisbytes = 1;
	    sval = i2c_byte_read( node, thiscmd, thisdata );
	}
	else 
	{
	    if ( bytes_remaining <= 32 )
		thisbytes = bytes_remaining;
	    else
		thisbytes = 32;		/* MAX SMBus transfer block */

	    sval = i2c_block_read( node, thiscmd, thisbytes, thisdata );
	}
#else
	/* Bodge: The above block read/write function doesn't actually work
	 * so all bytes get sent as 1-byte transactions */
	thisbytes = 1;
	sval = i2c_byte_read( node, thiscmd, thisdata );
#endif

	if( sval != STATUS_SUCCESS )
	{
	    retries++;
	    if( retries > MAX_RETRY_COUNT )
	    {
		mobo_logf( LOG_CRIT
		    "I2C: failure on reading %d bytes from address 0x%02X\n",
		    bytes, node );
		return STATUS_FAILURE;
	    }

	    continue;		/* Go round again without incrementing */
	}

	/* Otherwise we're ready to handle the next chunk of data */
	thiscmd += thisbytes;
	thisdata += thisbytes;
	bytes_remaining -= thisbytes;
	retries = 0;
    }

    return STATUS_SUCCESS;
}

#if 0
    /* ALI problem - a byte can get duplicated and read the next transaction */
    if ( last_byte_read != data[0] )
    {
	last_byte_read = (unsigned)data[0];
    } else {
	mobo_logf( LOG_WARN 
		"I2C: duplicated byte on read from device at 0x%02X\n",
		node );
	
	data[0] = SMB_READ( SMBUS_DATA_A );
    }
#endif



/* node = I2C bus address of target
 * cmd = goes into SMBus command byte, often used by targets as a reg index 
 *      (set to -1 if not required)
 * bytes = number of bytes to be written to target
 * data = location of data to write
 */

DBM_STATUS i2cdev_write( const uint8 node, const int cmd,
			 const size_t bytes, const uint8 *data)
{
    int bytes_remaining, retries;
    uint8 const *thisdata;
    int thiscmd;
    int thisbytes;
    DBM_STATUS sval;

    TRACE( "I2C bus write of %d bytes to node %x cmd %02x\n",
		bytes, node, cmd  );

    bytes_remaining = bytes;
    thisdata = data;
    thiscmd = cmd;
    thisbytes = 0;
    retries = 0;

    while( bytes_remaining > 0 )
    {
#if 0
	if( bytes_remaining == 1 )
	{
	    thisbytes = 1;
	    sval = i2c_byte_write( node, thiscmd, thisdata );
	}
	else 
	{
	    if ( bytes_remaining <= 32 )
		thisbytes = bytes_remaining;
	    else
		thisbytes = 32;		/* MAX SMBus transfer block */

	    sval = i2c_block_write( node, thiscmd, thisbytes, thisdata );
	}
#else
	/* Bodge: The above block transfer may/may not work so currently
 	 * each byte is sent individually */
	thisbytes = 1;
	sval = i2c_byte_write( node, thiscmd, thisdata );
#endif

	if( sval != STATUS_SUCCESS )
	{
	    retries++;
	    if( retries > MAX_RETRY_COUNT )
	    {
		mobo_logf( LOG_CRIT
		    "I2C: failure on writing %d bytes to address 0x%02X\n",
		    bytes, node );
		return STATUS_FAILURE;
	    }

	    continue;		/* Go round again without incrementing */
	}

	/* Otherwise we're ready to handle the next chunk of data */
	thiscmd += thisbytes;
	thisdata += thisbytes;
 	bytes_remaining -= thisbytes;
	retries = 0;
    }

    return STATUS_SUCCESS;
}

